// AI Generator Global Functions
class AIGenerator {
    constructor() {
        this.baseUrl = window.location.origin;
        this.csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
    }

    // Generic AI generation function
    async generateContent(endpoint, data, targetElementId, statusElementId, buttonId) {
        console.log('AI Generator Debug:', { endpoint, data, targetElementId, statusElementId, buttonId });
        
        const button = document.getElementById(buttonId);
        const icon = button?.querySelector('i');
        const text = button?.querySelector('span');
        const status = document.getElementById(statusElementId);
        const targetElement = document.getElementById(targetElementId);

        if (!button || !targetElement) {
            console.error('Missing elements:', { button, targetElement });
            this.showAlert('Error: Elemen form tidak ditemukan!', 'danger');
            return;
        }

        // Validation
        if (!this.validateRequiredFields(data)) {
            this.showAlert('Harap isi field yang diperlukan terlebih dahulu!', 'warning');
            return;
        }

        // Show loading state
        this.setLoadingState(button, icon, text, status, true);

        try {
            // Prepare form data
            const formData = new FormData();
            Object.keys(data).forEach(key => {
                if (data[key]) {
                    formData.append(key, data[key]);
                    console.log('Added to FormData:', key, data[key]);
                }
            });

            // Add CSRF token if available
            if (this.csrfToken) {
                formData.append('csrf_token', this.csrfToken);
            }

            const apiUrl = `${this.baseUrl}/api/ai/${endpoint}`;
            console.log('Calling API:', apiUrl);

            // Call AI API
            const response = await fetch(apiUrl, {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });

            console.log('Response status:', response.status);
            const result = await response.json();
            console.log('API Result:', result);

            if (result.success) {
                targetElement.value = result.text;
                this.showAlert('Konten berhasil digenerate oleh AI!', 'success');
                this.updateStatus(statusElementId, 'Berhasil digenerate!', 'success');
                
                // Auto hide status after 3 seconds
                setTimeout(() => {
                    if (status) status.style.display = 'none';
                }, 3000);

                // Trigger change event for form validation
                targetElement.dispatchEvent(new Event('change'));
            } else {
                this.showAlert('Gagal generate konten: ' + (result.message || 'Unknown error'), 'danger');
                this.updateStatus(statusElementId, 'Gagal generate: ' + (result.message || 'Unknown error'), 'error');
            }
        } catch (error) {
            console.error('AI Generation Error:', error);
            this.showAlert('Terjadi kesalahan: ' + error.message, 'danger');
            this.updateStatus(statusElementId, 'Terjadi kesalahan!', 'error');
        } finally {
            // Reset button state
            this.setLoadingState(button, icon, text, status, false);
        }
    }

    // Set loading state for button and status
    setLoadingState(button, icon, text, status, isLoading) {
        if (!button) return;
        
        if (isLoading) {
            button.disabled = true;
            if (icon) icon.className = 'spinner-border spinner-border-sm';
            if (text) text.textContent = 'Tunggu sebentar yahh..';
            if (status) {
                status.style.display = 'block';
                status.className = 'ai-status mt-2';
            }
        } else {
            button.disabled = false;
            if (icon) icon.className = 'bi bi-magic';
            if (text) text.textContent = 'Generate AI';
        }
    }

    // Update status message
    updateStatus(statusElementId, message, type = 'info') {
        const statusText = document.querySelector(`#${statusElementId} .alert span`);
        const statusContainer = document.getElementById(statusElementId);
        
        if (statusText) {
            statusText.textContent = message;
        }
        
        if (statusContainer) {
            statusContainer.className = `ai-status mt-2 ${type}`;
        }
    }

    // Validate required fields - check if at least one key field is filled
    validateRequiredFields(data) {
        // Check if any of the main fields have data
        const hasData = Object.values(data).some(value => value && value.trim() !== '');
        return hasData;
    }

    // Show alert notification
    showAlert(message, type = 'info') {
        // Remove existing alerts
        const existingAlerts = document.querySelectorAll('.alert.ai-notification');
        existingAlerts.forEach(alert => alert.remove());

        const alertDiv = document.createElement('div');
        alertDiv.className = `alert alert-${type} alert-dismissible fade show ai-notification`;
        alertDiv.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 1060;
            min-width: 300px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        `;
        
        alertDiv.innerHTML = `
            <i class="bi bi-${this.getAlertIcon(type)} me-2"></i>
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        
        document.body.appendChild(alertDiv);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (alertDiv.parentNode) {
                alertDiv.remove();
            }
        }, 5000);
    }

    // Get appropriate icon for alert type
    getAlertIcon(type) {
        const icons = {
            'success': 'check-circle',
            'danger': 'exclamation-triangle',
            'warning': 'exclamation-triangle',
            'info': 'info-circle'
        };
        return icons[type] || 'info-circle';
    }

    // Specific generators for different content types
    async generateEducationDescription() {
        const data = {
            nama_sekolah: document.getElementById('nama_sekolah')?.value || '',
            jurusan: document.getElementById('jurusan')?.value || '',
            jenjang: document.getElementById('jenjang')?.value || '',
            tahun_mulai: document.getElementById('tahun_mulai')?.value || '',
            tahun_selesai: document.getElementById('tahun_selesai')?.value || ''
        };

        await this.generateContent(
            'generate-education',
            data,
            'deskripsi',
            'aiStatus',
            'generateDescriptionBtn'
        );
    }

    async generateWorkDescription() {
        const data = {
            nama_perusahaan: document.getElementById('nama_perusahaan')?.value || '',
            posisi: document.getElementById('posisi')?.value || '',
            tahun_mulai: document.getElementById('tahun_mulai')?.value || '',
            tahun_selesai: document.getElementById('tahun_selesai')?.value || ''
        };

        await this.generateContent(
            'generate-work',
            data,
            'deskripsi',
            'aiStatus',
            'generateDescriptionBtn'
        );
    }

    async generateProjectDescription() {
        const data = {
            nama_project: document.getElementById('nama_project')?.value || '',
            teknologi: document.getElementById('teknologi')?.value || '',
            jenis_project: document.getElementById('jenis_project')?.value || ''
        };

        await this.generateContent(
            'generate-project',
            data,
            'deskripsi',
            'aiStatus',
            'generateDescriptionBtn'
        );
    }

    async generateTrainingDescription() {
        const data = {
            nama_training: document.getElementById('nama_pelatihan')?.value || '',
            penyelenggara: document.getElementById('penyelenggara')?.value || '',
            durasi: this.calculateDuration()
        };

        await this.generateContent(
            'generate-training',
            data,
            'deskripsi',
            'aiStatus',
            'generateDescriptionBtn'
        );
    }

    async generateOrganizationDescription() {
        const data = {
            nama_organisasi: document.getElementById('nama_organisasi')?.value || '',
            jabatan: document.getElementById('jabatan')?.value || '',
            bidang: 'Organisasi Mahasiswa'
        };

        await this.generateContent(
            'generate-organization',
            data,
            'deskripsi',
            'aiStatus',
            'generateDescriptionBtn'
        );
    }

    async generateActivityDescription() {
        const data = {
            nama_aktivitas: document.getElementById('nama_aktivitas')?.value || '',
            lokasi: 'Kegiatan Harian',
            tipe_aktivitas: 'Personal Activity'
        };

        await this.generateContent(
            'generate-activity',
            data,
            'deskripsi',
            'aiStatus',
            'generateDescriptionBtn'
        );
    }

    async generateCertificateDescription() {
        const data = {
            nama_sertifikat: document.getElementById('nama_sertifikat')?.value || '',
            penerbit: document.getElementById('penerbit')?.value || '',
            bidang: 'Teknologi Informasi'
        };

        await this.generateContent(
            'generate-certificate',
            data,
            'deskripsi',
            'aiStatus',
            'generateDescriptionBtn'
        );
    }

    async generateBiodataProfile() {
        const data = {
            nama_lengkap: document.getElementById('nama_lengkap')?.value || '',
            keahlian: document.getElementById('keahlian')?.value || '',
            pendidikan: 'Mahasiswa Ilmu Komputer',
            pengalaman: 'Web Development, Mobile Development, AI'
        };

        await this.generateContent(
            'generate-biodata-profile',
            data,
            'tentang_saya',
            'aiProfileStatus',
            'generateProfileBtn'
        );
    }

    async generateBiodataDescription() {
        const data = {
            nama_lengkap: document.getElementById('nama_lengkap')?.value || '',
            keahlian: document.getElementById('keahlian')?.value || '',
            pendidikan: 'Mahasiswa Ilmu Komputer di Universitas Yatsi Madani',
            pengalaman: 'Freelance Editor, Guru Kejuruan TKJ, Web Development, Mobile Development',
            minat: 'Web Development, Mobile Development, AI, Machine Learning, UI/UX Design'
        };

        await this.generateContent(
            'generate-biodata-description',
            data,
            'deskripsi_lengkap',
            'aiDescStatus',
            'generateDescriptionBtn'
        );
    }

    // Helper function to calculate duration from date fields
    calculateDuration() {
        const startDate = document.getElementById('tanggal_mulai')?.value;
        const endDate = document.getElementById('tanggal_selesai')?.value;
        
        if (startDate && endDate) {
            const start = new Date(startDate);
            const end = new Date(endDate);
            const diffTime = Math.abs(end - start);
            const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
            
            if (diffDays < 30) {
                return `${diffDays} hari`;
            } else if (diffDays < 365) {
                const months = Math.ceil(diffDays / 30);
                return `${months} bulan`;
            } else {
                const years = Math.ceil(diffDays / 365);
                return `${years} tahun`;
            }
        }
        
        return '1 bulan';
    }
}

// Initialize AI Generator
const aiGenerator = new AIGenerator();

// Fallback text generator
function generateFallbackText(inputText, type = 'activity') {
    const templates = {
        activity: [
            `Aktivitas "${inputText}" merupakan kegiatan yang memberikan pengalaman berharga dan pembelajaran yang berguna untuk pengembangan diri. Melalui aktivitas ini, saya dapat mengembangkan berbagai keterampilan dan memperoleh wawasan baru yang bermanfaat untuk karir dan kehidupan sehari-hari.`,
            
            `Kegiatan "${inputText}" ini sangat menarik dan memberikan kesempatan untuk belajar hal-hal baru. Pengalaman yang diperoleh dari aktivitas ini dapat menjadi bekal yang berharga untuk menghadapi tantangan di masa depan dan mengembangkan potensi diri.`
        ],
        
        training: [
            `Pelatihan "${inputText}" memberikan pengetahuan dan keterampilan baru yang sangat bermanfaat untuk pengembangan karir. Materi yang diajarkan sangat relevan dengan kebutuhan industri saat ini dan dapat langsung diaplikasikan dalam pekerjaan sehari-hari.`,
            
            `Melalui pelatihan "${inputText}", saya memperoleh pemahaman mendalam tentang konsep-konsep penting dan teknik-teknik praktis yang dapat meningkatkan produktivitas dan kualitas kerja.`
        ],
        
        project: [
            `Proyek "${inputText}" dikembangkan dengan menggunakan teknologi terkini dan mengikuti best practices dalam pengembangan software. Proyek ini memiliki fitur-fitur yang user-friendly dan dapat memberikan solusi yang efektif untuk kebutuhan pengguna.`,
            
            `Dalam pengembangan "${inputText}", saya menerapkan metodologi yang terstruktur dan menggunakan tools modern untuk memastikan kualitas dan performa yang optimal.`
        ],
        
        organization: [
            `Pengalaman berorganisasi di "${inputText}" memberikan kesempatan untuk mengembangkan kemampuan leadership, teamwork, dan komunikasi. Melalui berbagai kegiatan organisasi, saya dapat berkontribusi positif dan belajar bekerja sama dalam tim.`,
            
            `Sebagai bagian dari "${inputText}", saya terlibat aktif dalam berbagai program dan kegiatan yang memberikan dampak positif bagi komunitas dan lingkungan sekitar.`
        ],
        
        certificate: [
            `Sertifikat "${inputText}" memvalidasi kompetensi dan pengetahuan yang telah diperoleh melalui proses pembelajaran dan evaluasi yang ketat. Sertifikat ini menunjukkan kemampuan dan kredibilitas dalam bidang yang relevan.`,
            
            `Pencapaian sertifikat "${inputText}" merupakan bukti komitmen terhadap pengembangan profesional dan peningkatan kualitas diri dalam bidang keahlian tertentu.`
        ]
    };
    
    const typeTemplates = templates[type] || templates.activity;
    return typeTemplates[Math.floor(Math.random() * typeTemplates.length)];
}

// Simple test function for debugging
window.testAI = async function() {
    console.log('Testing AI Generator...');
    
    const formData = new FormData();
    formData.append('nama_aktivitas', 'Test Activity');
    formData.append('lokasi', 'Test Location');
    formData.append('tipe_aktivitas', 'Test Type');
    
    try {
        const response = await fetch(window.location.origin + '/api/ai/generate-activity', {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        });
        
        const result = await response.json();
        console.log('Test Result:', result);
        
        if (result.success) {
            alert('AI Test Success: ' + result.text);
        } else {
            alert('AI Test Failed: ' + result.message);
        }
    } catch (error) {
        console.error('Test Error:', error);
        alert('Test Error: ' + error.message);
    }
};

// Mode switching functions
window.switchMode = function(mode) {
    const manualTab = document.getElementById('manualTab');
    const aiTab = document.getElementById('aiTab');
    const manualMode = document.getElementById('manualMode');
    const aiMode = document.getElementById('aiMode');
    
    if (mode === 'manual') {
        manualTab.classList.add('active');
        aiTab.classList.remove('active');
        manualMode.style.display = 'block';
        aiMode.style.display = 'none';
    } else {
        aiTab.classList.add('active');
        manualTab.classList.remove('active');
        aiMode.style.display = 'block';
        manualMode.style.display = 'none';
        
        // Auto-fill AI input with existing data
        autoFillAIInput();
    }
};

// Auto-fill AI input based on form data
window.autoFillAIInput = function() {
    const aiInput = document.getElementById('aiInput');
    if (!aiInput || aiInput.value) return;
    
    // Detect form type and auto-fill accordingly
    const namaAktivitas = document.getElementById('nama_aktivitas')?.value;
    const namaPelatihan = document.getElementById('nama_pelatihan')?.value;
    const namaProject = document.getElementById('nama_project')?.value;
    const namaOrganisasi = document.getElementById('nama_organisasi')?.value;
    const namaSertifikat = document.getElementById('nama_sertifikat')?.value;
    
    if (namaAktivitas) {
        aiInput.value = `Saya melakukan aktivitas "${namaAktivitas}". Tolong buatkan deskripsi yang menarik dan profesional tentang aktivitas ini.`;
    } else if (namaPelatihan) {
        aiInput.value = `Saya mengikuti pelatihan "${namaPelatihan}". Tolong buatkan deskripsi yang menjelaskan manfaat dan pembelajaran dari pelatihan ini.`;
    } else if (namaProject) {
        aiInput.value = `Saya mengerjakan proyek "${namaProject}". Tolong buatkan deskripsi yang menjelaskan fitur, teknologi, dan tujuan dari proyek ini.`;
    } else if (namaOrganisasi) {
        aiInput.value = `Saya terlibat dalam organisasi "${namaOrganisasi}". Tolong buatkan deskripsi tentang peran dan kontribusi saya dalam organisasi ini.`;
    } else if (namaSertifikat) {
        aiInput.value = `Saya memperoleh sertifikat "${namaSertifikat}". Tolong buatkan deskripsi tentang kompetensi dan manfaat dari sertifikat ini.`;
    }
};

// Use AI result
window.useAIResult = function() {
    const aiOutput = document.getElementById('aiOutput');
    const deskripsi = document.getElementById('deskripsi');
    
    if (aiOutput.value && deskripsi) {
        deskripsi.value = aiOutput.value;
        // Switch back to manual mode
        switchMode('manual');
        alert('Hasil AI berhasil digunakan! Anda bisa mengeditnya di mode Manual.');
    }
};

// Regenerate AI - detect current form type
window.regenerateAI = function() {
    // Detect which form we're on and call appropriate function
    if (document.getElementById('nama_sertifikat')) {
        generateCertificateDescription();
    } else if (document.getElementById('nama_aktivitas')) {
        generateActivityDescription();
    } else if (document.getElementById('nama_pelatihan')) {
        generateTrainingDescription();
    } else if (document.getElementById('nama_project')) {
        generateProjectDescription();
    } else if (document.getElementById('nama_organisasi')) {
        generateOrganizationDescription();
    } else {
        generateActivityDescription(); // fallback
    }
};

// Enhanced AI generation function with new UI
window.generateActivityDescription = async function() {
    console.log('=== AI Generator Debug ===');
    
    // Get input from AI input field or form field
    const aiInput = document.getElementById('aiInput');
    const namaAktivitas = document.getElementById('nama_aktivitas')?.value || '';
    const inputText = aiInput?.value || namaAktivitas;
    
    const generateBtn = document.getElementById('generateBtn');
    const aiOutput = document.getElementById('aiOutput');
    const useResultBtn = document.getElementById('useResultBtn');
    const regenerateBtn = document.getElementById('regenerateBtn');
    
    console.log('Elements found:', {
        inputText: inputText,
        generateBtn: !!generateBtn,
        aiOutput: !!aiOutput
    });
    
    if (!inputText.trim()) {
        alert('Harap isi input untuk AI terlebih dahulu!');
        return;
    }
    
    console.log('Starting AI generation...');
    
    // Set loading state
    if (generateBtn) {
        generateBtn.disabled = true;
        generateBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-1"></span>Tunggu sebentar yahh..';
    }
    
    if (aiOutput) {
        aiOutput.classList.add('ai-loading');
        aiOutput.value = 'Sedang menggenerate...';
    }
    
    try {
        const formData = new FormData();
        formData.append('nama_aktivitas', inputText);
        formData.append('lokasi', 'Kegiatan Harian');
        formData.append('tipe_aktivitas', 'Personal Activity');
        
        console.log('Sending request to:', window.location.origin + '/api/ai/generate-activity');
        
        const response = await fetch(window.location.origin + '/api/ai/generate-activity', {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        });
        
        console.log('Response status:', response.status);
        const result = await response.json();
        console.log('API Result:', result);
        
        let generatedText = '';
        
        if (result.success && result.text && result.text.trim() !== '') {
            generatedText = result.text;
            console.log('Success: Got text from AI API');
        } else {
            console.log('Using fallback text');
            generatedText = generateFallbackText(inputText, 'activity');
        }
        
        if (aiOutput) {
            aiOutput.value = generatedText;
            aiOutput.classList.remove('ai-loading');
        }
        
        // Enable action buttons
        if (useResultBtn) useResultBtn.disabled = false;
        if (regenerateBtn) regenerateBtn.disabled = false;
        
        alert('Deskripsi berhasil digenerate! Lihat hasilnya di bagian "Hasil AI".');
        
    } catch (error) {
        console.error('Error:', error);
        
        if (aiOutput) {
            const fallbackText = generateFallbackText(inputText, 'activity');
            aiOutput.value = fallbackText;
            aiOutput.classList.remove('ai-loading');
        }
        
        // Enable action buttons
        if (useResultBtn) useResultBtn.disabled = false;
        if (regenerateBtn) regenerateBtn.disabled = false;
        
        alert('Terjadi kesalahan, menggunakan teks alternatif.');
    } finally {
        if (generateBtn) {
            generateBtn.disabled = false;
            generateBtn.innerHTML = '<i class="bi bi-magic me-1"></i>Generate AI';
        }
        console.log('=== End Debug ===');
    }
};

// Enhanced training description generator
window.generateTrainingDescription = async function() {
    const aiInput = document.getElementById('aiInput');
    const namaPelatihan = document.getElementById('nama_pelatihan')?.value || '';
    const inputText = aiInput?.value || namaPelatihan;
    
    await generateAIContentNew('generate-training', {
        nama_training: inputText,
        penyelenggara: document.getElementById('penyelenggara')?.value || '',
        durasi: '1 bulan'
    }, 'training');
};

window.generateProjectDescription = async function() {
    const aiInput = document.getElementById('aiInput');
    const namaProject = document.getElementById('nama_project')?.value || '';
    const inputText = aiInput?.value || namaProject;
    
    await generateAIContentNew('generate-project', {
        nama_project: inputText,
        teknologi: document.getElementById('teknologi')?.value || '',
        jenis_project: document.getElementById('jenis_project')?.value || ''
    }, 'project');
};

window.generateOrganizationDescription = async function() {
    const aiInput = document.getElementById('aiInput');
    const namaOrganisasi = document.getElementById('nama_organisasi')?.value || '';
    const inputText = aiInput?.value || namaOrganisasi;
    
    await generateAIContentNew('generate-organization', {
        nama_organisasi: inputText,
        jabatan: document.getElementById('jabatan')?.value || '',
        bidang: 'Organisasi Mahasiswa'
    }, 'organization');
};

window.generateCertificateDescription = async function() {
    console.log('=== Certificate AI Generator Called ===');
    
    const aiInput = document.getElementById('aiInput');
    const namaSertifikat = document.getElementById('nama_sertifikat')?.value || '';
    const penerbit = document.getElementById('penerbit')?.value || '';
    const inputText = aiInput?.value || namaSertifikat;
    
    console.log('Certificate elements:', {
        aiInput: !!aiInput,
        namaSertifikat: namaSertifikat,
        penerbit: penerbit,
        inputText: inputText
    });
    
    if (!inputText.trim()) {
        alert('Harap isi nama sertifikat atau input AI terlebih dahulu!');
        return;
    }
    
    const generateBtn = document.getElementById('generateBtn');
    const aiOutput = document.getElementById('aiOutput');
    const useResultBtn = document.getElementById('useResultBtn');
    const regenerateBtn = document.getElementById('regenerateBtn');
    
    // Set loading state
    if (generateBtn) {
        generateBtn.disabled = true;
        generateBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-1"></span>Tunggu sebentar yahh..';
    }
    
    if (aiOutput) {
        aiOutput.classList.add('ai-loading');
        aiOutput.value = 'Sedang menggenerate...';
    }
    
    try {
        const formData = new FormData();
        formData.append('nama_sertifikat', inputText);
        formData.append('penerbit', penerbit);
        formData.append('bidang', 'Teknologi Informasi');
        
        console.log('Sending request to:', window.location.origin + '/api/ai/generate-certificate');
        
        const response = await fetch(window.location.origin + '/api/ai/generate-certificate', {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        });
        
        console.log('Response status:', response.status);
        const result = await response.json();
        console.log('API Result:', result);
        
        let generatedText = '';
        
        if (result.success && result.text && result.text.trim() !== '') {
            generatedText = result.text;
            console.log('Success: Got text from AI API');
        } else {
            console.log('Using fallback text');
            generatedText = generateFallbackText(inputText, 'certificate');
        }
        
        if (aiOutput) {
            aiOutput.value = generatedText;
            aiOutput.classList.remove('ai-loading');
        }
        
        // Enable action buttons
        if (useResultBtn) useResultBtn.disabled = false;
        if (regenerateBtn) regenerateBtn.disabled = false;
        
        alert('Deskripsi berhasil digenerate! Lihat hasilnya di bagian "Hasil AI".');
        
    } catch (error) {
        console.error('Error:', error);
        
        if (aiOutput) {
            const fallbackText = generateFallbackText(inputText, 'certificate');
            aiOutput.value = fallbackText;
            aiOutput.classList.remove('ai-loading');
        }
        
        // Enable action buttons
        if (useResultBtn) useResultBtn.disabled = false;
        if (regenerateBtn) regenerateBtn.disabled = false;
        
        alert('Terjadi kesalahan, menggunakan teks alternatif.');
    } finally {
        if (generateBtn) {
            generateBtn.disabled = false;
            generateBtn.innerHTML = '<i class="bi bi-magic me-1"></i>Generate AI';
        }
        console.log('=== End Certificate AI Generator ===');
    }
};

// New generic function for enhanced AI UI
window.generateAIContentNew = async function(endpoint, data, contentType) {
    console.log('=== Enhanced AI Generator ===');
    
    const generateBtn = document.getElementById('generateBtn');
    const aiOutput = document.getElementById('aiOutput');
    const useResultBtn = document.getElementById('useResultBtn');
    const regenerateBtn = document.getElementById('regenerateBtn');
    
    // Get main input value
    const mainValue = Object.values(data).find(val => val && val.trim() !== '') || '';
    
    if (!mainValue.trim()) {
        alert('Harap isi input untuk AI terlebih dahulu!');
        return;
    }
    
    // Set loading state
    if (generateBtn) {
        generateBtn.disabled = true;
        generateBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-1"></span>Tunggu sebentar yahh..';
    }
    
    if (aiOutput) {
        aiOutput.classList.add('ai-loading');
        aiOutput.value = 'Sedang menggenerate...';
    }
    
    try {
        const formData = new FormData();
        Object.keys(data).forEach(key => {
            if (data[key]) formData.append(key, data[key]);
        });
        
        const response = await fetch(window.location.origin + '/api/ai/' + endpoint, {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        });
        
        const result = await response.json();
        console.log('API Result:', result);
        
        let generatedText = '';
        
        if (result.success && result.text && result.text.trim() !== '') {
            generatedText = result.text;
        } else {
            generatedText = generateFallbackText(mainValue, contentType);
        }
        
        if (aiOutput) {
            aiOutput.value = generatedText;
            aiOutput.classList.remove('ai-loading');
        }
        
        // Enable action buttons
        if (useResultBtn) useResultBtn.disabled = false;
        if (regenerateBtn) regenerateBtn.disabled = false;
        
        alert('Deskripsi berhasil digenerate! Lihat hasilnya di bagian "Hasil AI".');
        
    } catch (error) {
        console.error('Error:', error);
        
        if (aiOutput) {
            const fallbackText = generateFallbackText(mainValue, contentType);
            aiOutput.value = fallbackText;
            aiOutput.classList.remove('ai-loading');
        }
        
        if (useResultBtn) useResultBtn.disabled = false;
        if (regenerateBtn) regenerateBtn.disabled = false;
        
        alert('Terjadi kesalahan, menggunakan teks alternatif.');
    } finally {
        if (generateBtn) {
            generateBtn.disabled = false;
            generateBtn.innerHTML = '<i class="bi bi-magic me-1"></i>Generate AI';
        }
    }
};

// Generic function for all AI content generation (old version)
window.generateAIContent = async function(endpoint, data) {
    console.log('Generating AI content:', endpoint, data);
    
    const button = document.getElementById('generateDescriptionBtn');
    const textarea = document.getElementById('deskripsi');
    
    // Check if main field has data
    const hasData = Object.values(data).some(value => value && value.trim() !== '');
    if (!hasData) {
        alert('Harap isi field yang diperlukan terlebih dahulu!');
        return;
    }
    
    if (button) {
        button.disabled = true;
        button.innerHTML = '<span class="spinner-border spinner-border-sm me-1"></span>Tunggu sebentar yahh..';
    }
    
    try {
        const formData = new FormData();
        Object.keys(data).forEach(key => {
            if (data[key]) formData.append(key, data[key]);
        });
        
        const response = await fetch(window.location.origin + '/api/ai/' + endpoint, {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        });
        
        const result = await response.json();
        console.log('AI Result:', result);
        
        if (result.success && textarea) {
            if (result.text && result.text.trim() !== '') {
                textarea.value = result.text;
                alert('Deskripsi berhasil digenerate oleh AI!');
            } else {
                console.log('Warning: Empty text received, using fallback');
                // Detect content type and generate appropriate fallback
                let contentType = 'activity';
                let mainValue = 'konten';
                
                if (endpoint.includes('training')) {
                    contentType = 'training';
                    mainValue = data.nama_training || 'pelatihan';
                } else if (endpoint.includes('project')) {
                    contentType = 'project';
                    mainValue = data.nama_project || 'proyek';
                } else if (endpoint.includes('organization')) {
                    contentType = 'organization';
                    mainValue = data.nama_organisasi || 'organisasi';
                } else if (endpoint.includes('certificate')) {
                    contentType = 'certificate';
                    mainValue = data.nama_sertifikat || 'sertifikat';
                } else {
                    mainValue = Object.values(data).find(val => val && val.trim() !== '') || 'aktivitas';
                }
                
                const fallbackText = generateFallbackText(mainValue, contentType);
                textarea.value = fallbackText;
                alert('Deskripsi berhasil digenerate!');
            }
        } else {
            alert('Gagal generate: ' + (result.message || 'Unknown error'));
        }
    } catch (error) {
        console.error('Error:', error);
        alert('Terjadi kesalahan: ' + error.message);
    } finally {
        if (button) {
            button.disabled = false;
            button.innerHTML = '<i class="bi bi-magic"></i> <span>Generate AI</span>';
        }
    }
};

// Global functions for backward compatibility
window.generateEducationDescription = () => aiGenerator.generateEducationDescription();
window.generateWorkDescription = () => aiGenerator.generateWorkDescription();
window.generateBiodataProfile = () => aiGenerator.generateBiodataProfile();
window.generateBiodataDescription = () => aiGenerator.generateBiodataDescription();

// Ensure certificate function is globally available
if (typeof window.generateCertificateDescription === 'undefined') {
    window.generateCertificateDescription = async function() {
        console.log('Fallback certificate generator called');
        return aiGenerator.generateCertificateDescription();
    };
}

// Auto-setup AI generate buttons when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Add AI generate functionality to existing forms
    const aiButtons = document.querySelectorAll('[id*="generate"]');
    aiButtons.forEach(button => {
        if (!button.onclick) {
            const buttonId = button.id;
            if (buttonId.includes('Education')) {
                button.onclick = () => aiGenerator.generateEducationDescription();
            } else if (buttonId.includes('Work')) {
                button.onclick = () => aiGenerator.generateWorkDescription();
            } else if (buttonId.includes('Project')) {
                button.onclick = () => aiGenerator.generateProjectDescription();
            } else if (buttonId.includes('Training')) {
                button.onclick = () => aiGenerator.generateTrainingDescription();
            } else if (buttonId.includes('Organization')) {
                button.onclick = () => aiGenerator.generateOrganizationDescription();
            } else if (buttonId.includes('Activity')) {
                button.onclick = () => aiGenerator.generateActivityDescription();
            } else if (buttonId.includes('Certificate')) {
                button.onclick = () => aiGenerator.generateCertificateDescription();
            } else if (buttonId.includes('Profile')) {
                button.onclick = () => aiGenerator.generateBiodataProfile();
            } else if (buttonId.includes('Description')) {
                button.onclick = () => aiGenerator.generateBiodataDescription();
            }
        }
    });
});